# 機能設計書 23-結合層（Merge Layers）

## 概要

本ドキュメントは、TensorFlow/Kerasにおけるテンソル結合層（Merge Layers）の機能設計を記述する。Add・Subtract・Multiply・Average・Maximum・Minimum・Concatenate・Dotの各結合操作をレイヤーとして提供する。

### 本機能の処理概要

**業務上の目的・背景**：ニューラルネットワークにおいて、複数の処理パスからの出力を統合することは非常に一般的な要件である。Residual Connection（残差接続）にはAdd、Feature Fusion（特徴量融合）にはConcatenate、Attention WeightingにはMultiplyが使用される。これらの結合操作を標準レイヤーとして提供することで、Functional APIやSubclassed Modelでの複雑なネットワーク構築を容易にする。

**機能の利用シーン**：ResNet等のスキップ接続（Add）、マルチ入力モデルでの特徴量結合（Concatenate）、ゲート機構（Multiply）、アンサンブル平均（Average）、特徴量選択（Maximum/Minimum）などで使用される。

**主要な処理内容**：
1. 入力テンソルリストの形状検証とブロードキャスト互換性確認
2. 異なるランクの入力テンソルに対するリシェイプ処理
3. 要素ごとの結合演算（Add, Subtract, Multiply, Average, Maximum, Minimum）
4. 軸方向の結合（Concatenate）
5. ドット積による結合（Dot）
6. マスクの伝播処理

**関連システム・外部連携**：コア層（No.24）のDense等と組み合わせてFunctional APIモデルを構築する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

計算処理（テンソル結合演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| inputs | list[Tensor] | Yes | 結合する入力テンソルのリスト | 最低2つ、Subtractは正確に2つ |
| axis | int | No | Concatenateの結合軸 | デフォルト: -1 |
| axes | int/tuple | No | Dotの内積軸 | 必須（Dot使用時） |
| normalize | bool | No | Dotで正規化するか | デフォルト: False |

### 入力データソース

Kerasモデルの前段レイヤー出力。要素ごと操作では同一形状（ブロードキャスト可能）が必要。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | Tensor | 結合結果テンソル |

### 出力先

次段のKerasレイヤーまたはモデル出力。

## 処理フロー

### 処理シーケンス

```
1. ビルド（build）
   └─ 入力形状の検証、ブロードキャスト互換性確認
   └─ リシェイプ要否判定（_reshape_required）
2. 呼び出し（call）
   └─ 2.1 リシェイプが必要な場合の入力調整
   └─ 2.2 _merge_function呼び出し（サブクラスで実装）
   └─ 2.3 転置された場合の出力の逆転置
```

### フローチャート

```mermaid
flowchart TD
    A[入力テンソルリスト] --> B{リスト形式?}
    B -->|No| ERR[ValueError]
    B -->|Yes| C{長さ >= 2?}
    C -->|No| ERR
    C -->|Yes| D{リシェイプ必要?}
    D -->|Yes| E[ランク揃え処理]
    D -->|No| F[そのまま]
    E --> G[_merge_function呼び出し]
    F --> G
    G --> H[出力テンソル]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | 最低入力数 | 結合層は最低2つの入力が必要 | build/call時 |
| BR-23-02 | Subtract制約 | Subtractは正確に2入力のみ | build/call時 |
| BR-23-03 | バッチサイズ統一 | 全入力のバッチサイズが同一 | build時 |
| BR-23-04 | ブロードキャスト | 異なる形状はブロードキャスト可能であること | 要素ごと操作時 |

### 計算ロジック

- **Add**: `output = sum(inputs)`
- **Subtract**: `output = inputs[0] - inputs[1]`
- **Multiply**: `output = product(inputs)`
- **Average**: `output = sum(inputs) / len(inputs)`
- **Maximum**: `output = element-wise max(inputs)`
- **Minimum**: `output = element-wise min(inputs)`
- **Concatenate**: `output = concat(inputs, axis)`
- **Dot**: `output = batch_dot(inputs[0], inputs[1], axes)`

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | ビルドエラー | 入力がリスト形式でない | リスト形式で渡す |
| ValueError | ビルドエラー | 入力数が2未満 | 2つ以上の入力を提供する |
| ValueError | ビルドエラー | 異なるバッチサイズ | バッチサイズを統一する |
| ValueError | 形状エラー | ブロードキャスト不可能な形状 | 入力形状を確認する |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 全操作は要素ごとのテンソル演算で、GPU上で効率的に実行される
- リシェイプ処理は入力ランクが異なる場合のみ発生する

## セキュリティ考慮事項

特になし。

## 備考

- 関数形式のショートカット（tf.keras.layers.add, tf.keras.layers.concatenate等）も提供
- マスク伝播は全入力マスクのAND結合で計算される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: ベースクラスの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | merge.py | `tensorflow/python/keras/layers/merge.py` | _Mergeベースクラス（29-214行目）の構造を理解 |

**読解のコツ**: `_merge_function`は抽象メソッドで各サブクラスが実装する。`_compute_elemwise_op_output_shape`（47-83行目）でブロードキャスト互換性を検証する仕組みを把握すること。

#### Step 2: ビルドとコール処理

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | merge.py | `tensorflow/python/keras/layers/merge.py` | build（86-114行目）での形状検証とリシェイプ要否判定 |
| 2-2 | merge.py | `tensorflow/python/keras/layers/merge.py` | call（116-179行目）でのリシェイプとマージ処理 |

**主要処理フロー**:
1. **86-98行目**: 入力形状検証（リスト形式、最低2入力、バッチサイズ統一）
2. **111-114行目**: _reshape_requiredフラグの設定
3. **119-132行目**: ランクが既知の場合、expand_dimsで揃える
4. **134-177行目**: ランクが未知の場合、転置→マージ→逆転置

#### Step 3: 各結合演算の実装

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | merge.py | `tensorflow/python/keras/layers/merge.py` | Add（217-250行目）, Subtract（253-288行目）, Multiply（291-317行目） |
| 3-2 | merge.py | `tensorflow/python/keras/layers/merge.py` | Average（320-353行目）, Maximum（356-382行目）, Minimum（385行目以降） |

#### Step 4: マスク伝播

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | merge.py | `tensorflow/python/keras/layers/merge.py` | compute_mask（200-214行目）の全マスクAND結合ロジック |

### プログラム呼び出し階層図

```
tf.keras.layers.Add / Subtract / Multiply / ...
    │
    ├─ _Merge (base class)
    │      ├─ build()
    │      │    └─ _compute_elemwise_op_output_shape()
    │      ├─ call()
    │      │    ├─ expand_dims (ランク揃え)
    │      │    └─ _merge_function() [各サブクラスで実装]
    │      └─ compute_mask()
    │
    ├─ Add._merge_function: output += inputs[i]
    ├─ Subtract._merge_function: inputs[0] - inputs[1]
    ├─ Multiply._merge_function: output *= inputs[i]
    ├─ Average._merge_function: sum / len
    ├─ Maximum._merge_function: math_ops.maximum
    └─ Minimum._merge_function: math_ops.minimum
```

### データフロー図

```
[入力]                      [処理]                       [出力]

Tensor A [B,D]     ───▶  _Merge.call()           ───▶  結合結果 [B,D]
Tensor B [B,D]     ───▶    ├─ 形状検証
                            ├─ リシェイプ(必要時)
                            └─ _merge_function()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| merge.py | `tensorflow/python/keras/layers/merge.py` | ソース | 全結合レイヤーの定義 |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス |
| array_ops.py | `tensorflow/python/ops/array_ops.py` | ソース | expand_dims, transpose, reshape |
| math_ops.py | `tensorflow/python/ops/math_ops.py` | ソース | maximum, minimum等の数学演算 |
